package com.ustadmobile.core.db.dao

import androidx.lifecycle.LiveData
import com.ustadmobile.door.attachments.storeAttachment
import com.ustadmobile.door.ext.doorPrimaryKeyManager
import com.ustadmobile.door.ext.rootDatabase
import com.ustadmobile.door.room.RoomDatabase
import com.ustadmobile.door.util.systemTimeInMillis
import com.ustadmobile.lib.db.entities.PersonPicture
import com.ustadmobile.lib.db.entities.asEntityWithAttachment
import kotlin.Long
import kotlin.Unit
import kotlin.collections.List
import kotlinx.coroutines.runBlocking

public class PersonPictureDao_ReplicateWrapper(
  private val _db: RoomDatabase,
  private val _dao: PersonPictureDao,
) : PersonPictureDao() {
  public override suspend fun replicateOnNewNode(newNodeId: Long): Unit {
    _dao.replicateOnNewNode(newNodeId)
  }

  public override suspend fun replicateOnChange(): Unit {
    _dao.replicateOnChange()
  }

  public override suspend fun findByPersonUidAsync(personUid: Long): PersonPicture? =
      _dao.findByPersonUidAsync(personUid)

  public override fun findByPersonUidLive(personUid: Long): LiveData<PersonPicture?> =
      _dao.findByPersonUidLive(personUid)

  public override suspend fun updateAsync(personPicture: PersonPicture): Unit {
    _db.storeAttachment(personPicture.asEntityWithAttachment())
    personPicture.personPictureLct = systemTimeInMillis()
    //must set versionid and/or primary key here
    _dao.updateAsync(personPicture)
  }

  public override fun insert(entity: PersonPicture): Long {
    runBlocking {
      _db.storeAttachment(entity.asEntityWithAttachment())
    }
    val _pkManager = _db.rootDatabase.doorPrimaryKeyManager
    if(entity.personPictureUid == 0L) {
      val _newPk = _pkManager.nextId(50)
      entity.personPictureUid = _newPk
    }
    entity.personPictureLct = systemTimeInMillis()
    //must set versionid and/or primary key here
    _dao.insert(entity)
    return entity.personPictureUid
  }

  public override suspend fun insertAsync(entity: PersonPicture): Long {
    _db.storeAttachment(entity.asEntityWithAttachment())
    val _pkManager = _db.rootDatabase.doorPrimaryKeyManager
    if(entity.personPictureUid == 0L) {
      val _newPk = _pkManager.nextIdAsync(50)
      entity.personPictureUid = _newPk
    }
    entity.personPictureLct = systemTimeInMillis()
    //must set versionid and/or primary key here
    _dao.insertAsync(entity)
    return entity.personPictureUid
  }

  public override fun insertList(entityList: List<PersonPicture>): Unit {
    runBlocking {
      entityList.forEach {
        _db.storeAttachment(it.asEntityWithAttachment())
      }
    }
    val _pkManager = _db.rootDatabase.doorPrimaryKeyManager
    val _generatedPks = mutableListOf<Long>()
    entityList.iterator().forEach  {
      if(it.personPictureUid == 0L) {
        val _newPk = _pkManager.nextId(50)
        it.personPictureUid = _newPk
        _generatedPks += _newPk
      }
      it.personPictureLct = systemTimeInMillis()
    }
    //must set versionid and/or primary key here
    _dao.insertList(entityList)
  }

  public override fun updateList(entityList: List<PersonPicture>): Unit {
    runBlocking {
      entityList.forEach {
        _db.storeAttachment(it.asEntityWithAttachment())
      }
    }
    val _generatedPks = mutableListOf<Long>()
    entityList.iterator().forEach  {
      it.personPictureLct = systemTimeInMillis()
    }
    //must set versionid and/or primary key here
    _dao.updateList(entityList)
  }

  public override fun update(entity: PersonPicture): Unit {
    runBlocking {
      _db.storeAttachment(entity.asEntityWithAttachment())
    }
    entity.personPictureLct = systemTimeInMillis()
    //must set versionid and/or primary key here
    _dao.update(entity)
  }
}
