package com.ustadmobile.core.db.dao

import androidx.paging.PagingSource
import androidx.room.Dao
import androidx.room.Insert
import androidx.room.Query
import androidx.room.Update
import com.ustadmobile.lib.db.composites.ContentEntryAndDetail
import com.ustadmobile.lib.db.composites.ContentEntryAndLanguage
import com.ustadmobile.lib.db.composites.ContentEntryAndListDetail
import com.ustadmobile.lib.db.composites.ContentEntryAndPicture
import com.ustadmobile.lib.db.entities.ContentEntry
import com.ustadmobile.lib.db.entities.ContentEntryWithLanguage
import com.ustadmobile.lib.db.entities.UidAndLabel
import kotlin.Any
import kotlin.Boolean
import kotlin.Int
import kotlin.Long
import kotlin.String
import kotlin.collections.List
import kotlinx.coroutines.flow.Flow

@Dao
public actual abstract class ContentEntryDao : BaseDao<ContentEntry> {
  @Insert(
    entity = Any::class,
    onConflict = 3,
  )
  public actual abstract suspend fun insertListAsync(entityList: List<ContentEntry>)

  @Insert(
    onConflict = 1,
    entity = Any::class,
  )
  public actual abstract suspend fun upsertAsync(entity: ContentEntry)

  @Query(`value` = "SELECT * FROM ContentEntry WHERE contentEntryUid = :entryUid")
  public actual abstract suspend fun findByUidAsync(entryUid: Long): ContentEntry?

  @Query(`value` =
      "\n        SELECT ContentEntry.*, Language.* \n          FROM ContentEntry \n               LEFT JOIN Language \n                         ON Language.langUid = ContentEntry.primaryLanguageUid\n         WHERE ContentEntry.contentEntryUid=:entryUuid\n        ")
  public actual abstract suspend fun findEntryWithLanguageByEntryIdAsync(entryUuid: Long):
      ContentEntryAndLanguage?

  @Query(`value` =
      "\n        SELECT ContentEntry.*, ContentEntryPicture2.*\n          FROM ContentEntry\n               LEFT JOIN ContentEntryPicture2 \n                         ON ContentEntryPicture2.cepUid = :uid\n         WHERE ContentEntry.contentEntryUid = :uid                \n    ")
  public actual abstract suspend fun findByUidWithEditDetails(uid: Long): ContentEntryAndPicture?

  @Query(`value` =
      "\n              -- When the user is viewing ContentEntryDetail where the class is specified eg \n              -- for a ContentEntry that is part of a Clazz then results information will only be\n              -- included if the user is a student in the class\n              -- If the user is viewing the ContentEntryDetail via the library then the results\n              -- information will always be included\n              WITH IncludeResults(includeResults) AS (\n                   SELECT CAST(\n                      (SELECT (:clazzUid = 0)\n                           OR (\n        (SELECT EXISTS(\n                SELECT 1\n                  FROM ClazzEnrolment\n                 WHERE ClazzEnrolment.clazzEnrolmentClazzUid = :clazzUid\n                   AND ClazzEnrolment.clazzEnrolmentPersonUid = :accountPersonUid\n                   AND ClazzEnrolment.clazzEnrolmentRole = 1000))\n    )\n                      ) AS INTEGER)\n                  )\n\n              SELECT ContentEntry.*, ContentEntryVersion.*, ContentEntryPicture2.*,\n                   :accountPersonUid AS sPersonUid,\n                   :courseBlockUid AS sCbUid,\n                   \n                   (SELECT MAX(StatementEntity.extensionProgress)\n                     FROM StatementEntity\n                    WHERE (SELECT includeResults FROM IncludeResults) = 1\n                      AND StatementEntity.statementActorPersonUid = :accountPersonUid\n                      AND StatementEntity.statementContentEntryUid = ContentEntry.contentEntryUid\n                      AND CAST(StatementEntity.completionOrProgress AS INTEGER) = 1\n                   ) AS sProgress,\n                   (SELECT CASE\n                       -- If a successful completion statement exists, then count as success\n                       WHEN (SELECT EXISTS(\n                                    SELECT 1\n                                      FROM StatementEntity\n                                     WHERE (SELECT includeResults FROM IncludeResults) = 1\n                                       AND StatementEntity.statementActorPersonUid = :accountPersonUid\n                                       AND StatementEntity.statementContentEntryUid = ContentEntry.contentEntryUid \n                                       AND (\n              CAST(StatementEntity.completionOrProgress AS INTEGER) = 1\n          AND CAST(StatementEntity.resultCompletion AS INTEGER) = 1    \n          AND CAST(StatementEntity.resultSuccess AS INTEGER) = 1\n    )))\n                            THEN 1\n                       -- Else if no success record exists, however a fail record exists, mark as failed\n                       WHEN (SELECT EXISTS(\n                                    SELECT 1\n                                      FROM StatementEntity\n                                     WHERE (SELECT includeResults FROM IncludeResults) = 1\n                                       AND StatementEntity.statementActorPersonUid = :accountPersonUid\n                                       AND StatementEntity.statementContentEntryUid = ContentEntry.contentEntryUid \n                                       AND (\n              CAST(StatementEntity.completionOrProgress AS INTEGER) = 1\n          AND CAST(StatementEntity.resultCompletion AS INTEGER) = 1\n          AND CAST(StatementEntity.resultSuccess AS INTEGER) = 0\n    )))\n                            THEN 0\n                            \n                       ELSE NULL\n                       END                    \n                   ) AS sIsSuccess,\n                   (SELECT EXISTS(\n                           SELECT 1\n                             FROM StatementEntity\n                            WHERE (SELECT includeResults FROM IncludeResults) = 1\n                              AND StatementEntity.statementActorPersonUid = :accountPersonUid\n                              AND StatementEntity.statementContentEntryUid = ContentEntry.contentEntryUid\n                              AND CAST(StatementEntity.resultCompletion AS INTEGER) = 1\n                              AND CAST(StatementEntity.completionOrProgress AS INTEGER) = 1)\n                   ) AS sIsCompleted,\n                   (SELECT MAX(StatementEntity.resultScoreScaled)\n                      FROM StatementEntity\n                     WHERE (SELECT includeResults FROM IncludeResults) = 1\n                       AND StatementEntity.statementActorPersonUid = :accountPersonUid\n                       AND StatementEntity.statementContentEntryUid = ContentEntry.contentEntryUid\n                   ) AS sScoreScaled\n    \n              FROM ContentEntry\n                   LEFT JOIN ContentEntryVersion\n                             ON ContentEntryVersion.cevUid = \n                             (SELECT ContentEntryVersion.cevUid\n                                FROM ContentEntryVersion\n                               WHERE ContentEntryVersion.cevContentEntryUid = :contentEntryUid\n                                 AND CAST(cevInActive AS INTEGER) = 0\n                            ORDER BY ContentEntryVersion.cevLct DESC\n                              LIMIT 1)\n                   LEFT JOIN ContentEntryPicture2\n                             ON ContentEntryPicture2.cepUid = :contentEntryUid   \n             WHERE ContentEntry.contentEntryUid = :contentEntryUid\n            ")
  public actual abstract fun findByContentEntryUidWithDetailsAsFlow(
    contentEntryUid: Long,
    clazzUid: Long,
    courseBlockUid: Long,
    accountPersonUid: Long,
  ): Flow<ContentEntryAndDetail?>

  @Query(`value` = "SELECT * FROM ContentEntry WHERE sourceUrl = :sourceUrl LIMIT 1")
  public actual abstract fun findBySourceUrl(sourceUrl: String): ContentEntry?

  @Query(`value` = "SELECT title FROM ContentEntry WHERE contentEntryUid = :entryUid")
  public actual abstract suspend fun findTitleByUidAsync(entryUid: Long): String?

  @Query(`value` =
      "SELECT ContentEntry.* FROM ContentEntry LEFT Join ContentEntryParentChildJoin ON ContentEntryParentChildJoin.cepcjChildContentEntryUid = ContentEntry.contentEntryUid WHERE ContentEntryParentChildJoin.cepcjParentContentEntryUid = :parentUid")
  public actual abstract fun getChildrenByParentUid(parentUid: Long):
      PagingSource<Int, ContentEntry>

  @Query(`value` =
      "\n        SELECT ContentEntry.*\n          FROM ContentEntryParentChildJoin\n               JOIN ContentEntry \n                    ON ContentEntryParentChildJoin.cepcjChildContentEntryUid = ContentEntry.contentEntryUid\n         WHERE ContentEntryParentChildJoin.cepcjParentContentEntryUid = :parentUid\n    ")
  public actual abstract suspend fun getChildrenByParentAsync(parentUid: Long): List<ContentEntry>

  @Query(`value` =
      "SELECT COUNT(*) FROM ContentEntry LEFT Join ContentEntryParentChildJoin ON ContentEntryParentChildJoin.cepcjChildContentEntryUid = ContentEntry.contentEntryUid WHERE ContentEntryParentChildJoin.cepcjParentContentEntryUid = :parentUid")
  public actual abstract suspend fun getCountNumberOfChildrenByParentUUidAsync(parentUid: Long): Int

  @Query(`value` = "SELECT * FROM ContentEntry where contentEntryUid = :parentUid LIMIT 1")
  public actual abstract suspend fun getContentByUuidAsync(parentUid: Long): ContentEntry?

  @Query(`value` =
      "SELECT ContentEntry.* FROM ContentEntry LEFT JOIN ContentEntryRelatedEntryJoin ON ContentEntryRelatedEntryJoin.cerejRelatedEntryUid = ContentEntry.contentEntryUid WHERE ContentEntryRelatedEntryJoin.relType = 1 AND ContentEntryRelatedEntryJoin.cerejRelatedEntryUid != :entryUuid")
  public actual abstract suspend fun findAllLanguageRelatedEntriesAsync(entryUuid: Long):
      List<ContentEntry>

  @Update(
    entity = Any::class,
    onConflict = 3,
  )
  actual abstract override fun update(entity: ContentEntry)

  @Query(`value` =
      "\n        SELECT ContentEntry.*, Language.*\n          FROM ContentEntry\n               LEFT JOIN Language \n                      ON Language.langUid = ContentEntry.primaryLanguageUid \n         WHERE ContentEntry.contentEntryUid = :uid              \n    ")
  public actual abstract suspend fun findByUidWithLanguageAsync(uid: Long):
      ContentEntryWithLanguage?

  @Query(`value` = "SELECT * FROM ContentEntry WHERE contentEntryUid = :entryUid")
  public actual abstract fun findByUid(entryUid: Long): ContentEntry?

  @Query(`value` = "SELECT * FROM ContentEntry WHERE title = :title")
  public actual abstract fun findByTitle(title: String): Flow<ContentEntry?>

  @Query(`value` =
      "SELECT ContentEntry.* FROM ContentEntry WHERE ContentEntry.sourceUrl = :sourceUrl")
  public actual abstract suspend fun findBySourceUrlWithContentEntryStatusAsync(sourceUrl: String):
      ContentEntry?

  @Query(`value` =
      "\n            WITH IncludeResults(includeResults) AS (SELECT 1)\n            \n            SELECT ContentEntry.*, ContentEntryParentChildJoin.*, ContentEntryPicture2.*,\n                   \n               :accountPersonUid AS sPersonUid,\n               0 AS sCbUid,\n               \n                   (SELECT MAX(StatementEntity.extensionProgress)\n                     FROM StatementEntity\n                    WHERE (SELECT includeResults FROM IncludeResults) = 1\n                      AND StatementEntity.statementActorPersonUid = :accountPersonUid\n                      AND StatementEntity.statementContentEntryUid = ContentEntry.contentEntryUid\n                      AND CAST(StatementEntity.completionOrProgress AS INTEGER) = 1\n                   ) AS sProgress,\n                   (SELECT CASE\n                       -- If a successful completion statement exists, then count as success\n                       WHEN (SELECT EXISTS(\n                                    SELECT 1\n                                      FROM StatementEntity\n                                     WHERE (SELECT includeResults FROM IncludeResults) = 1\n                                       AND StatementEntity.statementActorPersonUid = :accountPersonUid\n                                       AND StatementEntity.statementContentEntryUid = ContentEntry.contentEntryUid \n                                       AND (\n              CAST(StatementEntity.completionOrProgress AS INTEGER) = 1\n          AND CAST(StatementEntity.resultCompletion AS INTEGER) = 1    \n          AND CAST(StatementEntity.resultSuccess AS INTEGER) = 1\n    )))\n                            THEN 1\n                       -- Else if no success record exists, however a fail record exists, mark as failed\n                       WHEN (SELECT EXISTS(\n                                    SELECT 1\n                                      FROM StatementEntity\n                                     WHERE (SELECT includeResults FROM IncludeResults) = 1\n                                       AND StatementEntity.statementActorPersonUid = :accountPersonUid\n                                       AND StatementEntity.statementContentEntryUid = ContentEntry.contentEntryUid \n                                       AND (\n              CAST(StatementEntity.completionOrProgress AS INTEGER) = 1\n          AND CAST(StatementEntity.resultCompletion AS INTEGER) = 1\n          AND CAST(StatementEntity.resultSuccess AS INTEGER) = 0\n    )))\n                            THEN 0\n                            \n                       ELSE NULL\n                       END                    \n                   ) AS sIsSuccess,\n                   (SELECT EXISTS(\n                           SELECT 1\n                             FROM StatementEntity\n                            WHERE (SELECT includeResults FROM IncludeResults) = 1\n                              AND StatementEntity.statementActorPersonUid = :accountPersonUid\n                              AND StatementEntity.statementContentEntryUid = ContentEntry.contentEntryUid\n                              AND CAST(StatementEntity.resultCompletion AS INTEGER) = 1\n                              AND CAST(StatementEntity.completionOrProgress AS INTEGER) = 1)\n                   ) AS sIsCompleted,\n                   (SELECT MAX(StatementEntity.resultScoreScaled)\n                      FROM StatementEntity\n                     WHERE (SELECT includeResults FROM IncludeResults) = 1\n                       AND StatementEntity.statementActorPersonUid = :accountPersonUid\n                       AND StatementEntity.statementContentEntryUid = ContentEntry.contentEntryUid\n                   ) AS sScoreScaled\n    \n    \n              FROM ContentEntry \n                    LEFT JOIN ContentEntryParentChildJoin \n                         ON ContentEntryParentChildJoin.cepcjChildContentEntryUid = ContentEntry.contentEntryUid \n                    LEFT JOIN ContentEntryPicture2\n                         ON ContentEntryPicture2.cepUid = ContentEntry.contentEntryUid\n             WHERE ContentEntryParentChildJoin.cepcjParentContentEntryUid = :parentUid \n               AND (:langParam = 0 OR ContentEntry.primaryLanguageUid = :langParam)\n               AND (:categoryParam0 = 0 OR :categoryParam0 \n                    IN (SELECT ceccjContentCategoryUid \n                          FROM ContentEntryContentCategoryJoin \n                         WHERE ceccjContentEntryUid = ContentEntry.contentEntryUid)) \n               AND (CAST(:includeDeleted AS INTEGER) = 1 OR CAST(ContentEntryParentChildJoin.cepcjDeleted AS INTEGER) = 0)          \n            ORDER BY ContentEntryParentChildJoin.childIndex,\n                     CASE(:sortOrder)\n                     WHEN 1 THEN ContentEntry.title\n                     ELSE ''\n                     END ASC,\n                     CASE(:sortOrder)\n                     WHEN 2 THEN ContentEntry.title\n                     ELSE ''\n                     END DESC,             \n                     ContentEntry.contentEntryUid")
  public actual abstract fun getChildrenByParentUidWithCategoryFilterOrderByName(
    accountPersonUid: Long,
    parentUid: Long,
    langParam: Long,
    categoryParam0: Long,
    sortOrder: Int,
    includeDeleted: Boolean,
  ): PagingSource<Int, ContentEntryAndListDetail>

  @Query(`value` =
      "\n        WITH IncludeResults(includeResults) AS (SELECT 1)\n        \n        SELECT ContentEntry.*, ContentEntryParentChildJoin.*, ContentEntryPicture2.*,\n               \n               :accountPersonUid AS sPersonUid,\n               0 AS sCbUid,\n               \n                   (SELECT MAX(StatementEntity.extensionProgress)\n                     FROM StatementEntity\n                    WHERE (SELECT includeResults FROM IncludeResults) = 1\n                      AND StatementEntity.statementActorPersonUid = :accountPersonUid\n                      AND StatementEntity.statementContentEntryUid = ContentEntry.contentEntryUid\n                      AND CAST(StatementEntity.completionOrProgress AS INTEGER) = 1\n                   ) AS sProgress,\n                   (SELECT CASE\n                       -- If a successful completion statement exists, then count as success\n                       WHEN (SELECT EXISTS(\n                                    SELECT 1\n                                      FROM StatementEntity\n                                     WHERE (SELECT includeResults FROM IncludeResults) = 1\n                                       AND StatementEntity.statementActorPersonUid = :accountPersonUid\n                                       AND StatementEntity.statementContentEntryUid = ContentEntry.contentEntryUid \n                                       AND (\n              CAST(StatementEntity.completionOrProgress AS INTEGER) = 1\n          AND CAST(StatementEntity.resultCompletion AS INTEGER) = 1    \n          AND CAST(StatementEntity.resultSuccess AS INTEGER) = 1\n    )))\n                            THEN 1\n                       -- Else if no success record exists, however a fail record exists, mark as failed\n                       WHEN (SELECT EXISTS(\n                                    SELECT 1\n                                      FROM StatementEntity\n                                     WHERE (SELECT includeResults FROM IncludeResults) = 1\n                                       AND StatementEntity.statementActorPersonUid = :accountPersonUid\n                                       AND StatementEntity.statementContentEntryUid = ContentEntry.contentEntryUid \n                                       AND (\n              CAST(StatementEntity.completionOrProgress AS INTEGER) = 1\n          AND CAST(StatementEntity.resultCompletion AS INTEGER) = 1\n          AND CAST(StatementEntity.resultSuccess AS INTEGER) = 0\n    )))\n                            THEN 0\n                            \n                       ELSE NULL\n                       END                    \n                   ) AS sIsSuccess,\n                   (SELECT EXISTS(\n                           SELECT 1\n                             FROM StatementEntity\n                            WHERE (SELECT includeResults FROM IncludeResults) = 1\n                              AND StatementEntity.statementActorPersonUid = :accountPersonUid\n                              AND StatementEntity.statementContentEntryUid = ContentEntry.contentEntryUid\n                              AND CAST(StatementEntity.resultCompletion AS INTEGER) = 1\n                              AND CAST(StatementEntity.completionOrProgress AS INTEGER) = 1)\n                   ) AS sIsCompleted,\n                   (SELECT MAX(StatementEntity.resultScoreScaled)\n                      FROM StatementEntity\n                     WHERE (SELECT includeResults FROM IncludeResults) = 1\n                       AND StatementEntity.statementActorPersonUid = :accountPersonUid\n                       AND StatementEntity.statementContentEntryUid = ContentEntry.contentEntryUid\n                   ) AS sScoreScaled\n    \n    \n          FROM CourseBlock\n               JOIN ContentEntry \n                    ON CourseBlock.cbType = 104\n                       AND ContentEntry.contentEntryUid = CourseBlock.cbEntityUid\n                       AND CAST(CourseBlock.cbActive AS INTEGER) = 1\n               LEFT JOIN ContentEntryParentChildJoin\n                         ON ContentEntryParentChildJoin.cepcjParentContentEntryUid = 0\n               LEFT JOIN ContentEntryPicture2\n                         ON ContentEntryPicture2.cepUid = ContentEntry.contentEntryUid          \n         WHERE CourseBlock.cbClazzUid IN\n               (SELECT ClazzEnrolment.clazzEnrolmentClazzUid\n                  FROM ClazzEnrolment\n                 WHERE ClazzEnrolment.clazzEnrolmentPersonUid = :accountPersonUid)\n    ")
  public actual abstract fun getContentFromMyCourses(accountPersonUid: Long):
      PagingSource<Int, ContentEntryAndListDetail>

  @Query(`value` =
      "\n        WITH IncludeResults(includeResults) AS (SELECT 1)\n        \n        SELECT ContentEntry.*, ContentEntryParentChildJoin.*, ContentEntryPicture2.*, \n               \n               :accountPersonUid AS sPersonUid,\n               0 AS sCbUid,\n               \n                   (SELECT MAX(StatementEntity.extensionProgress)\n                     FROM StatementEntity\n                    WHERE (SELECT includeResults FROM IncludeResults) = 1\n                      AND StatementEntity.statementActorPersonUid = :accountPersonUid\n                      AND StatementEntity.statementContentEntryUid = ContentEntry.contentEntryUid\n                      AND CAST(StatementEntity.completionOrProgress AS INTEGER) = 1\n                   ) AS sProgress,\n                   (SELECT CASE\n                       -- If a successful completion statement exists, then count as success\n                       WHEN (SELECT EXISTS(\n                                    SELECT 1\n                                      FROM StatementEntity\n                                     WHERE (SELECT includeResults FROM IncludeResults) = 1\n                                       AND StatementEntity.statementActorPersonUid = :accountPersonUid\n                                       AND StatementEntity.statementContentEntryUid = ContentEntry.contentEntryUid \n                                       AND (\n              CAST(StatementEntity.completionOrProgress AS INTEGER) = 1\n          AND CAST(StatementEntity.resultCompletion AS INTEGER) = 1    \n          AND CAST(StatementEntity.resultSuccess AS INTEGER) = 1\n    )))\n                            THEN 1\n                       -- Else if no success record exists, however a fail record exists, mark as failed\n                       WHEN (SELECT EXISTS(\n                                    SELECT 1\n                                      FROM StatementEntity\n                                     WHERE (SELECT includeResults FROM IncludeResults) = 1\n                                       AND StatementEntity.statementActorPersonUid = :accountPersonUid\n                                       AND StatementEntity.statementContentEntryUid = ContentEntry.contentEntryUid \n                                       AND (\n              CAST(StatementEntity.completionOrProgress AS INTEGER) = 1\n          AND CAST(StatementEntity.resultCompletion AS INTEGER) = 1\n          AND CAST(StatementEntity.resultSuccess AS INTEGER) = 0\n    )))\n                            THEN 0\n                            \n                       ELSE NULL\n                       END                    \n                   ) AS sIsSuccess,\n                   (SELECT EXISTS(\n                           SELECT 1\n                             FROM StatementEntity\n                            WHERE (SELECT includeResults FROM IncludeResults) = 1\n                              AND StatementEntity.statementActorPersonUid = :accountPersonUid\n                              AND StatementEntity.statementContentEntryUid = ContentEntry.contentEntryUid\n                              AND CAST(StatementEntity.resultCompletion AS INTEGER) = 1\n                              AND CAST(StatementEntity.completionOrProgress AS INTEGER) = 1)\n                   ) AS sIsCompleted,\n                   (SELECT MAX(StatementEntity.resultScoreScaled)\n                      FROM StatementEntity\n                     WHERE (SELECT includeResults FROM IncludeResults) = 1\n                       AND StatementEntity.statementActorPersonUid = :accountPersonUid\n                       AND StatementEntity.statementContentEntryUid = ContentEntry.contentEntryUid\n                   ) AS sScoreScaled\n    \n    \n          FROM ContentEntry\n               LEFT JOIN ContentEntryParentChildJoin\n                         ON ContentEntryParentChildJoin.cepcjParentContentEntryUid = 0\n               LEFT JOIN ContentEntryPicture2\n                         ON ContentEntryPicture2.cepUid = ContentEntry.contentEntryUid\n         WHERE ContentEntry.contentOwner = :accountPersonUid\n    ")
  public actual abstract fun getContentByOwner(accountPersonUid: Long):
      PagingSource<Int, ContentEntryAndListDetail>

  @Update(
    entity = Any::class,
    onConflict = 3,
  )
  public actual abstract suspend fun updateAsync(entity: ContentEntry): Int

  @Query(`value` =
      "SELECT ContentEntry.* FROM ContentEntry LEFT JOIN ContentEntryParentChildJoin ON ContentEntryParentChildJoin.cepcjChildContentEntryUid = ContentEntry.contentEntryUid WHERE ContentEntryParentChildJoin.cepcjParentContentEntryUid = :parentUid")
  public actual abstract fun getChildrenByAll(parentUid: Long): List<ContentEntry>

  @Query(`value` = "SELECT * FROM ContentEntry where contentEntryUid = :parentUid LIMIT 1")
  public actual abstract fun findLiveContentEntry(parentUid: Long): Flow<ContentEntry?>

  @Query(`value` =
      "SELECT COALESCE((SELECT contentEntryUid \n                                      FROM ContentEntry \n                                     WHERE entryId = :objectId \n                                     LIMIT 1),0) AS ID")
  public actual abstract fun getContentEntryUidFromXapiObjectId(objectId: String): Long

  @Query(`value` = "SELECT * FROM ContentEntry WHERE sourceUrl LIKE :sourceUrl")
  public actual abstract fun findSimilarIdEntryForKhan(sourceUrl: String): List<ContentEntry>

  @Query(`value` =
      "\n            UPDATE ContentEntry \n               SET ceInactive = :ceInactive,\n                   contentEntryLct = :changedTime        \n            WHERE ContentEntry.contentEntryUid = :contentEntryUid")
  public actual abstract fun updateContentEntryInActive(
    contentEntryUid: Long,
    ceInactive: Boolean,
    changedTime: Long,
  )

  @Query(`value` =
      "\n        UPDATE ContentEntry \n           SET contentTypeFlag = :contentFlag,\n               contentEntryLct = :changedTime \n         WHERE ContentEntry.contentEntryUid = :contentEntryUid")
  public actual abstract fun updateContentEntryContentFlag(
    contentFlag: Int,
    contentEntryUid: Long,
    changedTime: Long,
  )

  @Query(`value` =
      "Select ContentEntry.contentEntryUid AS uid, ContentEntry.title As labelName \n                    from ContentEntry WHERE contentEntryUid IN (:contentEntryUids)")
  public actual abstract suspend fun getContentEntryFromUids(contentEntryUids: List<Long>):
      List<UidAndLabel>
}
