"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getStore = exports.getDatabase = exports.overrideAs = exports.setDatabaseInternal = void 0;
const helpers_1 = require("./helpers");
const STORES = new Map();
/**
 * When we save a database with an alias, it saves the actual version that database was opened with.
 * This prevents the `createObjectStore` method to create more than one store on any given database.
 * That is why we set the database instance internally every time we create a database version update.
 * When we do a version update, we don't have the database alias, only the database name. These two
 * Map objects help resolve the proper database using an alias and internally a database name.
 */
const DATABASES = new Map();
const DATABASE_ALIASES = new Map();
function setDatabaseInternal(databaseName, database) {
    DATABASES.set(databaseName, database);
}
exports.setDatabaseInternal = setDatabaseInternal;
function overrideAs(originalAs, alias, options) {
    const asArgs = [alias, options];
    // eslint-disable-next-line @typescript-eslint/ban-ts-comment
    // @ts-ignore
    const innerFn = originalAs.apply(this, asArgs);
    return (subject) => {
        if ((0, helpers_1.isIDBObjectStore)(subject)) {
            STORES.set(alias, subject);
            return subject;
        }
        else if ((0, helpers_1.isIDBDatabase)(subject)) {
            DATABASE_ALIASES.set(alias, subject.name);
            DATABASES.set(subject.name, subject);
            return subject;
        }
        else {
            return innerFn(subject);
        }
    };
}
exports.overrideAs = overrideAs;
exports.getDatabase = getIDBItem('database');
exports.getStore = getIDBItem('store');
function getIDBItem(type) {
    const map = type === 'store' ? STORES : DATABASE_ALIASES;
    return (alias) => {
        let error;
        const log = Cypress.log({
            autoEnd: false,
            type: 'parent',
            name: 'get',
            message: alias,
            consoleProps: () => ({
                error: error || 'no',
            }),
        });
        const withoutAtSign = alias.substring(1);
        if (map.has(withoutAtSign)) {
            log.end();
            const result = map.get(withoutAtSign);
            if (typeof result === 'string') {
                return cy.wrap(DATABASES.get(result));
            }
            return cy.wrap(result);
        }
        else {
            error = new Error(`could not find ${type} with alias ${alias}`);
            log.error(error).end();
            throw error;
        }
    };
}
//# sourceMappingURL=alias-setup.js.map