"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.createObjectStore = void 0;
const helpers_1 = require("./helpers");
const upgrade_database_1 = require("./upgrade-database");
function createObjectStoreInternal(database, storeName, options, isExistingStore) {
    let store;
    if (isExistingStore) {
        store = database.transaction(storeName, 'readwrite').objectStore(storeName);
    }
    else {
        store = options
            ? database.createObjectStore(storeName, options)
            : database.createObjectStore(storeName);
    }
    return new Promise((resolve, reject) => {
        store.transaction.oncomplete = () => {
            database.close();
            resolve(store);
        };
        store.transaction.onerror = (e) => {
            reject(e);
        };
    });
}
/**
 * creates an object store.
 * In order to create an object store, first, you need to initiate a database connection by calling the `cy.openIndexedDb('databaseName')` command and use the `as` chainer to store it with an alias.
 *
 * @example
 * You can chain off the `createObjectStore('storeName')` method from methods that yield an `IDBDatabase` instance (`openIndexedDb` or `getIndexedDb`). You can use the `as` chainer to save the store using an alias.
 * cy.getIndexedDb('@database').createObjectStore('example_store').as('exampleStore');
 *
 * You can also pass an optional options parameter to configure your object store. For example, you can create an object store with `autoIncrement` with the following command:
 * cy.getIndexedDb('@database')
 *  .createObjectStore('example_autoincrement_store', { autoIncrement: true })
 *  .as('exampleAutoincrementStore')
 *
 * @remarks You can retrieve the saved object store using the `cy.getStore('@exampleStore')`
 *
 * @param existingDatabase - `IDBDatabase` instance
 * @param storeName - Store name
 * @param options - `IDBObjectStoreParameters` interface
 * @returns Promise<IDBObjectStore>
 */
function createObjectStore(existingDatabase, storeName, options) {
    let error;
    const isExistingStore = existingDatabase.objectStoreNames.contains(storeName);
    const log = Cypress.log({
        name: `create`,
        type: 'child',
        message: `IDBObjectStore - ${storeName}`,
        consoleProps: () => ({
            'store name': storeName,
            'store creation options': options || 'default',
            'did store exist before?': isExistingStore,
            'database name': existingDatabase.name,
            error: error || 'no',
        }),
        autoEnd: false,
    });
    if (!(0, helpers_1.isIDBDatabase)(existingDatabase)) {
        error = new Error(`You tried to use the 'getObjectStore' method without calling 'openIndexedDb' first`);
        log.error(error).end();
        throw error;
    }
    const objectStoreDb = isExistingStore
        ? Promise.resolve(existingDatabase)
        : (0, upgrade_database_1.createVersionUpdateDatabaseConnection)(existingDatabase);
    return cy.wrap(objectStoreDb
        .then((versionUpdateDatabase) => createObjectStoreInternal(versionUpdateDatabase, storeName, options || null, isExistingStore))
        .then((store) => {
        log.end();
        return store;
    })
        .catch((e) => {
        error = e;
        log.error(e).end();
        throw e;
    }));
}
exports.createObjectStore = createObjectStore;
//# sourceMappingURL=create-object-store.js.map