"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.entries = exports.keys = void 0;
const helpers_1 = require("./helpers");
const open_database_1 = require("./open-database");
/**
 * List keys in store
 *
 * @remarks The `keys` method lists the keys available in the provided store.
 *
 * @param store `IDBObjectStore` instance
 *
 * @returns Promise<IDBValidKey[]>
 * @throws {Error} If it is chained off from a method that does not return an object store.
 */
function keys(store) {
    const { log, consoleProps } = createMetadataLog('keys');
    if (!(0, helpers_1.isIDBObjectStore)(store)) {
        const error = new Error(`You tried to use the 'keys' method without calling 'getObjectStore' first`);
        consoleProps.error = error;
        log.error(error).end();
        throw error;
    }
    return cy.wrap((0, open_database_1.createDatabaseConnection)(store.transaction.db.name)
        .then((openDb) => getMetadata(openDb, store, 'getAllKeys'))
        .then((result) => {
        consoleProps.result = result;
        log.end();
        return result;
    })
        .catch((e) => {
        consoleProps.error = e;
        log.error(e).end();
        throw e;
    }));
}
exports.keys = keys;
/**
 * List all entries in store
 *
 * @remarks The `entries` method lists all the values saved in the store.
 *
 * @param store `IDBObjectStore` instance
 *
 * @returns Promise<T[]>
 * @throws {Error} If it is chained off from a method that does not return an object store.
 */
function entries(store) {
    const { log, consoleProps } = createMetadataLog('entries');
    if (!(0, helpers_1.isIDBObjectStore)(store)) {
        const error = new Error(`You tried to use the 'entries' method without calling 'getObjectStore' first`);
        consoleProps.error = error;
        log.error(error).end();
        throw error;
    }
    return cy.wrap((0, open_database_1.createDatabaseConnection)(store.transaction.db.name)
        .then((openDb) => getMetadata(openDb, store, 'getAll'))
        .then((result) => {
        consoleProps.result = result;
        log.end();
        return result;
    })
        .catch((e) => {
        consoleProps.error = e;
        log.error(e).end();
        throw e;
    }));
}
exports.entries = entries;
function getMetadata(db, store, operation) {
    return new Promise((resolve, reject) => {
        const request = db
            .transaction(store.name, 'readwrite')
            .objectStore(store.name)[operation]();
        request.onerror = (e) => {
            db.close();
            reject(e);
        };
        request.onsuccess = () => {
            request.onerror = () => void 0;
            db.close();
            const result = request.result;
            resolve(result);
        };
    });
}
function createMetadataLog(operation) {
    const consoleProps = {};
    const log = Cypress.log({
        autoEnd: false,
        type: 'child',
        name: operation,
        message: `IDBObjectStore ${operation}`,
        consoleProps: () => consoleProps,
    });
    return { log, consoleProps };
}
//# sourceMappingURL=object-store-metadata.js.map