"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.createVersionUpdateDatabaseConnection = void 0;
const rxjs_1 = require("rxjs");
const alias_setup_1 = require("./alias-setup");
/**
 * Create a version update database connection
 *
 * @remarks This method should be used only internally. The method creates a database upgrade before creating a new object store.
 *
 * @param openDatabase `IDBDatabase` instance
 *
 * @returns IDBDatabase
 */
function createVersionUpdateDatabaseConnection(openDatabase) {
    let error;
    let databaseVersion;
    const databaseName = openDatabase.name;
    const log = Cypress.log({
        name: `upgrade`,
        type: 'parent',
        message: `IDBDatabase - ${databaseName}`,
        consoleProps: () => ({
            'old database version': openDatabase.version,
            'new database version': databaseVersion,
            'database name': databaseName,
            error: error || 'no',
        }),
        autoEnd: false,
    });
    return new Promise((resolve, reject) => {
        openDatabase.close();
        const request = window.indexedDB.open(databaseName, openDatabase.version + 1);
        request.onerror = (e) => {
            error = e;
            log.error(e).end();
            reject(e);
        };
        request.onupgradeneeded = (e) => {
            request.onerror = () => void 0;
            const db = e.target.result;
            db.onversionchange = rxjs_1.noop;
            databaseVersion = db.version;
            (0, alias_setup_1.setDatabaseInternal)(databaseName, db);
            log.end();
            resolve(db);
        };
    });
}
exports.createVersionUpdateDatabaseConnection = createVersionUpdateDatabaseConnection;
//# sourceMappingURL=upgrade-database.js.map