package com.ustadmobile.libuicompose.view.person.child

import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.material3.OutlinedTextField
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.testTag
import com.ustadmobile.core.MR
import com.ustadmobile.core.impl.UstadMobileConstants
import com.ustadmobile.core.viewmodel.person.child.EditChildProfileUiState
import com.ustadmobile.core.viewmodel.person.child.EditChildProfileViewModel
import com.ustadmobile.lib.db.entities.Person
import com.ustadmobile.lib.db.entities.ext.shallowCopy
import com.ustadmobile.libuicompose.components.UstadDateField
import com.ustadmobile.libuicompose.components.UstadMessageIdOptionExposedDropDownMenuField
import com.ustadmobile.libuicompose.components.UstadVerticalScrollColumn
import com.ustadmobile.libuicompose.util.ext.defaultItemPadding
import dev.icerock.moko.resources.compose.stringResource
import kotlinx.coroutines.Dispatchers
import moe.tlaster.precompose.flow.collectAsStateWithLifecycle

@Composable
fun EditChildProfileScreen(viewModel: EditChildProfileViewModel) {
    val uiState: EditChildProfileUiState by viewModel.uiState.collectAsStateWithLifecycle(
        EditChildProfileUiState(), Dispatchers.Main.immediate)
    EditChildProfileScreen(
        uiState = uiState,
        onPersonChanged = viewModel::onEntityChanged,
        onFullNameValueChange = viewModel::onFullNameValueChange,
    )
}

@Composable
fun EditChildProfileScreen(
    uiState: EditChildProfileUiState,
    onPersonChanged: (Person?) -> Unit = {},
    onFullNameValueChange: (String) -> Unit = { },
) {
    UstadVerticalScrollColumn(
        modifier = Modifier.fillMaxSize()
    )  {

        OutlinedTextField(
            modifier = Modifier
                .testTag("full_name")
                .fillMaxWidth()
                .defaultItemPadding(),
            value = uiState.fullName ?: "",
            label = { Text(stringResource(MR.strings.full_name) + "*") },
            isError = uiState.fullNameError != null,
            singleLine = true,
            onValueChange = { fullName ->
                onFullNameValueChange(fullName)
            },
            supportingText = {
                Text(uiState.fullNameError ?: stringResource(MR.strings.required))
            }
        )

        UstadMessageIdOptionExposedDropDownMenuField(
            value = uiState.person?.gender ?: 0,
            modifier = Modifier
                .testTag("gender")
                .defaultItemPadding()
                .fillMaxWidth(),
            label = stringResource(MR.strings.gender_literal) + "*",
            options = uiState.genderOptions.filter { it.stringResource != MR.strings.blank },
            onOptionSelected = {
                onPersonChanged(uiState.person?.shallowCopy{
                    gender = it.value
                })
            },
            isError = uiState.genderError != null,
            supportingText = {
                Text(uiState.genderError ?: stringResource(MR.strings.required))
            }
        )


            UstadDateField(
                modifier = Modifier.testTag("birthday").fillMaxWidth().defaultItemPadding(),
                value = uiState.person?.dateOfBirth ?: 0,
                label = { Text(stringResource(MR.strings.birthday)) },
                isError = uiState.dateOfBirthError != null,
                timeZoneId = UstadMobileConstants.UTC,
                onValueChange = {
                    onPersonChanged(uiState.person?.shallowCopy{
                        dateOfBirth = it
                    })
                },
                supportingText = uiState.dateOfBirthError?.let {
                    { Text(it) }
                }
            )


    }

}