"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.openIndexedDb = exports.createDatabaseConnection = void 0;
/**
 * Create a database connection
 *
 * @remarks The `createDatabaseConnection` opens a connection with the provided `databaseName` and `version`.
 *
 * @param databaseName Database name
 * @param versionConfiguredByUser Database version
 *
 * @returns Promise<IDBDatabase>
 * @throws {Error} If the connections fails to open.
 */
const rxjs_1 = require("rxjs");
function createDatabaseConnection(databaseName, versionConfiguredByUser) {
    const request = versionConfiguredByUser != null
        ? window.indexedDB.open(databaseName, versionConfiguredByUser)
        : window.indexedDB.open(databaseName);
    return new Promise((resolve, reject) => {
        request.onerror = (e) => {
            reject(e);
        };
        request.onsuccess = (e) => {
            request.onerror = () => void 0;
            const db = e.target.result;
            db.onversionchange = rxjs_1.noop;
            resolve(db);
        };
    });
}
exports.createDatabaseConnection = createDatabaseConnection;
/**
 * Open a database connection
 *
 * @remarks The `openIndexedDb` opens a connection with the provided `databaseName` and `version`.
 * @remarks the parameters is for developers who will touch this code in the future. It signals that this version number was passed down by the user and not incremented by our library.
 *
 * @param databaseName Database name
 * @param version Database version
 *
 * @returns Promise<IDBDatabase>
 * @throws {Error} If the connections fails to open.
 */
function openIndexedDb(databaseName, version) {
    let error;
    let databaseVersion;
    const log = Cypress.log({
        name: `open`,
        type: 'parent',
        message: `IDBDatabase - ${databaseName}`,
        consoleProps: () => ({
            'database version': databaseVersion,
            'database name': databaseName,
            error: error || 'no',
        }),
        autoEnd: false,
    });
    return cy.wrap(createDatabaseConnection(databaseName, version)
        .then((db) => {
        databaseVersion = db.version;
        log.end();
        return db;
    })
        .catch((e) => {
        error = e;
        log.error(e).end();
        throw e;
    }));
}
exports.openIndexedDb = openIndexedDb;
//# sourceMappingURL=open-database.js.map