package world.respect.credentials.passkey

import io.ktor.http.Url

/**
 * Data class representing the user handle for a passkey.
 *
 * As per the specs the user handle is specified by the relying party (e.g. RESPECT app) as an
 * identifier for a specific user account. It is an opaque byte sequence. It is encoded to base64
 * when creating a passkey and decoded from Base64 when verifying an authentication response for
 * sign-in.
 *
 *  a) The Person UID Number (64bit long UID - snowflake or GUID hash) - the UID number for the
 *     person in the database. As per https://w3c.github.io/webauthn/#user-handle :
 *
 *     "the user handle is chosen by the Relying Party and ought to be the same for all credentials
 *     registered to the same user account"
 *
 *  b) The School URL (encoded using string.toByteArray())
 *
 * As per https://w3c.github.io/webauthn/#user-handle and
 * https://w3c.github.io/webauthn/#dictionary-user-credential-params
 * the userHandle MUST NOT contain any personally identifiable information (like usernames, email,
 * phone etc).
 *
 * This user handle allows the app to verify the passkey on the server because it includes a) the
 * learning space url and b) the person UID. The server can find the passkey record in the database
 * by searching using the personUid AND credential id (generated by the provider). This is done in
 * VerifySignInWithPasskeyUseCase.
 *
 * @property personUidNum the Person UID number for this person (as per Person.personGuidHash)
 * @property schoolUrl the school url for this person
 */
data class RespectUserHandle(
    val personUidNum: Long,
    val schoolUrl: Url,
)