package com.ustadmobile.door.paging

import app.cash.paging.PagingSource
import app.cash.paging.PagingSourceLoadParams
import app.cash.paging.PagingSourceLoadResult
import app.cash.paging.PagingState
import com.ustadmobile.door.DoorDatabaseRepository
import com.ustadmobile.door.ext.DoorTag
import io.github.aakira.napier.Napier
import kotlinx.atomicfu.atomic
import kotlinx.coroutines.*
import kotlinx.coroutines.flow.update

/**
 * This is the primary offline-first PagingSource implementation - it is used for DAO functions that are annotated as
 * HttpAccessible that use the ClientStrategy PULL_REPLICATE_ENTITIES (which is used by default if the Strategy is set
 * to AUTO and the return type includes ReplicateEntity(s) via Embedded properties or inheritance).
 *
 * When a page is loaded a background http request will be made (including the PagingLoadParams) to the http endpoint.
 * Any ReplicateEntities that are received will be inserted/handled the normal way. This will trigger the underlying
 * dbPagingSource to invalidate if new entities are inserted.
 *
 * The implementation delegates to the dbPagingSource which is called immediately. This ensures that any entities in the
 * local database can be displayed immediately.
 *
 * @param repo The DoorDatabaseRepository beign used
 * @param repoPath the endpoint path (used for logging)
 * @param dbPagingSource the PagingSource from the underlying database
 * @param onLoadHttp a function (generated by the DoorRepositoryProcessor) that runs the http request and inserts any
 *        new replicate entities into the database.
 */
class DoorRepositoryReplicatePullPagingSource<Value: Any>(
    private val repo: DoorDatabaseRepository,
    private val repoPath: String,
    private val dbPagingSource: PagingSource<Int, Value>,
    private val onLoadHttp: suspend (params: PagingSourceLoadParams<Int>) -> Unit,
) : DoorRepositoryPagingSource<Int, Value>(){

    private val scope = CoroutineScope(Dispatchers.Default + Job())

    private val httpLoadCompletable = CompletableDeferred<Unit>()

    private val dbInvalidateCallbackRegistered = atomic(false)

    private val invalidated = atomic(false)

    private val onDbInvalidatedCallback: () -> Unit = {
        onDbInvalidated()
    }

    private fun onDbInvalidated() {
        Napier.v("DoorRepositoryReplicatePullPagingSource: onDbInvalidated")
        dbPagingSource.unregisterInvalidatedCallback(onDbInvalidatedCallback)
        if(!invalidated.getAndSet(true)) {
            scope.launch {
                try {
                    withTimeout(10000) {  httpLoadCompletable.await() }
                }catch(e: Exception) {
                    Napier.w("Exception waiting for httpLoad; $e")
                } finally {
                    Napier.v("DoorRepositoryReplicatePullPagingSource: call invalidate")
                    invalidate()
                    scope.cancel()
                }
            }
        }

    }

    override fun getRefreshKey(state: PagingState<Int, Value>): Int? {
        return dbPagingSource.getRefreshKey(state)
    }

    override suspend fun load(
        params: PagingSourceLoadParams<Int>
    ): PagingSourceLoadResult<Int, Value> {
        Napier.v("DoorRepositoryReplicatePullPagingSource: load key=${params.key}")
        if(!dbInvalidateCallbackRegistered.getAndSet(true)) {
            Napier.v("DoorRepositoryReplicatePullPagingSource: register db invalidate callback")
            dbPagingSource.registerInvalidatedCallback(onDbInvalidatedCallback)
        }

        scope.launch {
            val loadRequest = PagingSourceLoadState.PagingRequest(params.key)
            _loadState.update { prev ->
                prev.copyWithNewRequest(loadRequest)
            }
            try {
                onLoadHttp(params)
                httpLoadCompletable.complete(Unit)
                _loadState.update { prev ->
                    prev.copyWhenRequestCompleted(loadRequest)
                }
            }catch(e: Exception) {
                httpLoadCompletable.completeExceptionally(e)
                Napier.v(tag = DoorTag.LOG_TAG) { "" }
                _loadState.update { prev ->
                    prev.copyWhenRequestFailed(loadRequest)
                }
            }
        }

        return dbPagingSource.load(params)
    }

    companion object {

        const val PARAM_BATCHSIZE = "pagingBatchSize"

        const val PARAM_KEY = "pagingKey"

        const val PARAM_LOAD_PARAM_TYPE = "pagingLoadParamType"

    }
}